<?php
/**
 * Plugin Name:       Inomarker
 * Plugin URI:        https://inomarker.ru/widgets/wordpress
 * Description:       Автоматически маркирует упоминания иноагентов, террористов и экстремистов в тексте ваших статей.
 * Version:           1.1.4
 * Author:            Inomarker Team
 * Author URI:        https://inomarker.ru
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       inomarker
 *
 * @package           Inomarker
 */

// Защита от прямого доступа к файлу
if(!defined('ABSPATH')) {
    exit;
}

/**
 * Главный класс плагина, который инициализирует всю логику.
 * Мы используем шаблон Singleton, чтобы гарантировать только один экземпляр класса.
 */
final class Inomarker_Plugin
{

    /**
     * Версия плагина.
     *
     * @var string
     */
    public $version = '1.0.0';

    /**
     * Единственный экземпляр класса (Singleton).
     *
     * @var Inomarker_Plugin|null
     */
    private static $_instance = null;

    /**
     * Хранилище для сносок на время одной обработки.
     * @var array
     */
    private $footnotes_buffer = [];

    private $log_buffer = [];

    public function __construct() {
        // Регистрируем страницу в меню админки
        add_action('admin_menu', [
            $this,
            'add_admin_menu'
        ]);

        // Регистрируем настройки (поля)
        add_action('admin_init', [
            $this,
            'register_settings'
        ]);

        // Подключаем стили для админки (для красивого статуса)
        add_action('admin_enqueue_scripts', [
            $this,
            'enqueue_admin_styles'
        ]);

        // Это связывает нашу фоновую задачу (WP-Cron) с методом 'run_daily_sync'
        add_action('inomarker_daily_sync', [
            $this,
            'run_daily_sync'
        ]);

        add_filter('the_content', [
            $this,
            'mark_text_in_content'
        ],         10);

//        add_action('wp_enqueue_scripts', [
//            $this,
//            'enqueue_frontend_styles'
//        ]);

        add_action('admin_notices', [
            $this,
            'show_admin_notices'
        ]);

        add_action('save_post', [
            $this,
            'clear_post_cache_on_save'
        ]);

    }

    /**
     * Добавляет пункт меню "Inomarker" в "Настройки".
     */
    public function add_admin_menu() {
        add_options_page(
            'Настройки Inomarker', // Заголовок страницы
            'ИноМаркер',           // Название в меню
            'manage_options',      // Необходимые права (только админ)
            'inomarker-settings',  // Slug страницы
            [
                $this,
                'render_settings_page'
            ] // Функция, которая отрисует HTML
        );
    }

    /**
     * Регистрирует опции в базе данных и создает секции настроек.
     */
    public function register_settings() {
        // 1. API Ключ
        register_setting('inomarker_options_group', 'inomarker_api_key', [
            'type'              => 'string',
            'sanitize_callback' => [
                $this,
                'validate_and_save_api_key'
            ],
            'default'           => ''
        ]);

        // 2. Переключатель "Включено/Выключено"
        register_setting('inomarker_options_group', 'inomarker_enabled', [
            'type'              => 'boolean',
            'sanitize_callback' => [
                $this,
                'sanitize_bool_and_clear_cache'
            ],
            'default'           => false
        ]);

        // 3. Стиль маркировки
        register_setting('inomarker_options_group', 'inomarker_mark_style', [
            'type'              => 'string',
            'sanitize_callback' => [
                $this,
                'sanitize_string_and_clear_cache'
            ],
            'default'           => 'inline'
        ]);

        // 4. "Водный знак"
        register_setting('inomarker_options_group', 'inomarker_show_watermark', [
            'type'              => 'boolean',
            'sanitize_callback' => [
                $this,
                'sanitize_bool_and_clear_cache'
            ],
            'default'           => true,
        ]);

        // --- СЕКЦИЯ 1: Подключение ---
        add_settings_section(
            'inomarker_connection_section',
            'Подключение к сервису',
            null,
            'inomarker-settings'
        );

        add_settings_field(
            'inomarker_api_key',
            'API Ключ Виджета',
            [
                $this,
                'render_api_key_field'
            ],
            'inomarker-settings',
            'inomarker_connection_section'
        );

        // --- СЕКЦИЯ 2: Настройки отображения ---
        add_settings_section(
            'inomarker_display_section',
            'Настройки маркировки',
            null,
            'inomarker-settings'
        );

        add_settings_field(
            'inomarker_enabled',
            'Включить автомаркировку',
            [
                $this,
                'render_enabled_field'
            ],
            'inomarker-settings',
            'inomarker_display_section'
        );

        add_settings_field(
            'inomarker_mark_style',
            'Стиль маркировки',
            [
                $this,
                'render_style_field'
            ],
            'inomarker-settings',
            'inomarker_display_section'
        );

        add_settings_field(
            'inomarker_show_watermark',
            'Водный знак',
            [
                $this,
                'render_watermark_field'
            ],
            'inomarker-settings',
            'inomarker_display_section'
        );

    }

    /**
     * HTML для поля API ключа.
     */
    public function render_api_key_field() {
        $value = get_option('inomarker_api_key');
        echo '<input type="text" name="inomarker_api_key" value="' . esc_attr($value) . '" class="regular-text code">';
        echo '<p class="description">Скопируйте ключ из вашего <a href="https://inomarker.ru/widgets" target="_blank">личного кабинета</a></p>';
    }

    /**
     * HTML для чекбокса включения.
     */
    public function render_enabled_field() {
        $value = get_option('inomarker_enabled');
        $statusData = get_option('inomarker_subscription_status');

        $status = $statusData['status'] ?? 'unknown';
        $isDisabled = ($status !== 'active');

        echo '<label class="switch">';
        echo '<input type="checkbox" name="inomarker_enabled" value="1" ' . checked(1, $value, false) . ' ' . ($isDisabled ? 'disabled' : '') . '>';
        echo '<span class="slider round"></span>';
        echo '</label>';
        if($isDisabled) {
            echo '<p class="description" style="color: #d63638;">Требуется активная подписка для включения</p>';
        }
    }

    /**
     * HTML для выбора стиля.
     */
    public function render_style_field() {
        $value = get_option('inomarker_mark_style', 'inline');
        echo '<select name="inomarker_mark_style">';
        echo '<option value="inline" ' . selected($value, 'inline') . '>В тексте (в скобках)</option>';
        echo '<option value="footnote" ' . selected($value, 'footnote') . '>Сносками (*)</option>';
        echo '</select>';
        echo '<p class="description">';
        echo '<strong>В тексте:</strong> Иван Иванов (иноагент).<br>';
        echo '<strong>Сносками:</strong> Иван Иванов*. (Сноска будет добавлена в конец статьи)';
        echo '</p>';
    }

    /**
     * HTML для чекбокса "Водный знак".
     */
    public function render_watermark_field() {
        $value = get_option('inomarker_show_watermark', true);

        echo '<label class="switch">';
        echo '<input type="checkbox" name="inomarker_show_watermark" value="1" ' . checked(1, $value, false) . '>';
        echo '<span class="slider round"></span>';
        echo '</label>';
        echo '<p class="description">' . 'Добавляет в конец статьи строку "Маркировка произведена сервисом ИноМаркер".' . '</p>';
    }

    /**
     * Выполняется при АКТИВАЦИИ плагина.
     * Наша главная задача здесь - запланировать ежедневную фоновую задачу.
     */
    public static function on_activation() {
        if(!wp_next_scheduled('inomarker_daily_sync')) {
            wp_schedule_event(time(), 'daily', 'inomarker_daily_sync');
//            wp_schedule_event(time(), 'every_five_minutes', 'inomarker_daily_sync');
        }
    }

    /**
     * СКАЧИВАНИЕ РЕГУЛЯРОК
     * * Вызывается из 'run_daily_sync', если подписка активна.
     * Загружает data.json.
     *
     * @param string $api_key API-ключ для аутентификации.
     */
    private function fetch_regex_data($api_key) {
        // 1. Формируем URL для Эндпоинта 2
        $url = 'https://inomarker.ru/api/v1/plugin/regex-data';
        $request_url = add_query_arg('api_key', $api_key, $url);

        // 2. Делаем GET-запрос
        $response = wp_remote_get($request_url, [
            'timeout' => 30,
        ]);

        // 3. Обрабатываем ошибки
        if(is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            $error_message = is_wp_error($response) ? $response->get_error_message() : 'Не удалось получить данные с сервера.';
//            error_log('Inomarker Plugin Error (fetch_regex_data): ' . $error_message);
            return;
        }

        // 4. Получаем тело ответа (это наш JSON-файл в виде строки)
        $json_data = wp_remote_retrieve_body($response);

        // 5. Сохраняем JSON в папку /wp-content/uploads/
        $uploads = wp_upload_dir();
        $file_path = $uploads['basedir'] . '/inomarker-data.json';

        if(file_put_contents($file_path, $json_data) === false) {
//            error_log('Inomarker Plugin Error: Не удалось записать data.json в ' . $file_path);
        } else {
            // Записываем время успешного обновления, чтобы показать в админке
            update_option('inomarker_last_data_sync_time', time());
        }
    }

    /**
     * Подключает CSS для админки.
     */
    public function enqueue_admin_styles() {
        echo '<style>
            .inomarker-status-card {
                background: #fff; border: 1px solid #c3c4c7; padding: 20px; border-radius: 5px; margin-bottom: 20px; border-left: 4px solid #72aee6;
            }
            .inomarker-status-active { border-left-color: #00a32a; }
            .inomarker-status-inactive { border-left-color: #d63638; }
            .im-badge { padding: 5px 10px; border-radius: 4px; color: #fff; font-weight: bold; font-size: 12px; }
            .im-badge-green { background: #00a32a; }
            .im-badge-red { background: #d63638; }
            .im-badge-gray { background: #646970; }
            .switch { position: relative; display: inline-block; width: 60px; height: 34px; }
            .switch input { opacity: 0; width: 0; height: 0; }
            .slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: #ccc; transition: .4s; }
            .slider:before { position: absolute; content: ""; height: 26px; width: 26px; left: 4px; bottom: 4px; background-color: white; transition: .4s; }
            input:checked + .slider { background-color: #2196F3; }
            input:focus + .slider { box-shadow: 0 0 1px #2196F3; }
            input:checked + .slider:before { transform: translateX(26px); }
            .slider.round { border-radius: 34px; }
            .slider.round:before { border-radius: 50%; }
            .inomarker-notice p { padding-bottom: 5px; }
            .inomarker-notice .button-primary { font-weight: bold; }
        </style>';
    }

    /**
     * Подключает стили для маркировки на сам сайт (фронтенд).
     */
    public function enqueue_frontend_styles() {
        $css = "
            .inomarker-watermark {
                margin-top: 20px !important;
                padding-top: 20px !important;
                border-top: 1px solid #e5e7eb !important;
                font-size: 12px !important;
                color: #9ca3af !important;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
            }
            .inomarker-watermark a {
                color: #6b7280 !important;
                text-decoration: none !important;
                font-weight: 500;
            }
            .inomarker-watermark a:hover {
                text-decoration: underline !important;
            }
        ";
        wp_add_inline_style('wp-block-library', $css);
    }

    /**
     * Отрисовка всей страницы настроек.
     */
    public function render_settings_page() {
        // Получаем данные о статусе (которые мы будем обновлять через Cron)
        $subStatus = get_option('inomarker_subscription_status', ['status' => 'unknown']);
        $status = $subStatus['status'] ?? 'unknown';
        $tariff = $subStatus['tariff_name'] ?? 'Неизвестно';
        $expires = isset($subStatus['expires_at']) ? date_i18n('j F Y', strtotime($subStatus['expires_at'])) : '—';

        $is_plugin_enabled = get_option('inomarker_enabled', false);
        $lastSync = get_option('inomarker_last_sync_time') ? date_i18n('j F Y H:i', get_option('inomarker_last_sync_time')) : 'Никогда';
        $lastDataSync = get_option('inomarker_last_data_sync_time') ? date_i18n('j F Y H:i', get_option('inomarker_last_data_sync_time')) : 'Никогда';

        ?>
        <div class="wrap">
            <h1>ИноМаркер</h1>

            <div
                class="inomarker-status-card <?php echo ($status === 'active') ? 'inomarker-status-active' : 'inomarker-status-inactive'; ?>">
                <h2>Статус подписки</h2>
                <p>
                    <strong>Текущий статус:</strong>
                    <?php if($status === 'active'): ?>
                        <span class="im-badge im-badge-green">АКТИВНА</span>
                    <?php elseif($status === 'unknown'): ?>
                        <span class="im-badge im-badge-gray">НЕ ПРОВЕРЕНО</span>
                    <?php else: ?>
                        <span class="im-badge im-badge-red">НЕАКТИВНА</span>
                    <?php endif; ?>
                </p>
                <p>
                    <strong>Маркировка на сайте</strong>
                    <?php if($is_plugin_enabled && $status === 'active') : ?>
                        <span class="im-badge im-badge-green">ВКЛЮЧЕНА</span>
                    <?php else : ?>
                        <span class="im-badge im-badge-red">ВЫКЛЮЧЕНА</span>
                    <?php endif; ?>
                </p>
                <p><strong>Тариф:</strong> <?php echo esc_html($tariff); ?></p>
                <p><strong>Истекает:</strong> <?php echo esc_html($expires); ?></p>
                <hr>
                <p class="description">
                    <!--                    Последняя проверка статуса: -->
                    <?php //echo esc_html($lastSync); ?><!--<br>-->
                    Последнее обновление базы данных: <?php echo esc_html($lastDataSync); ?>
                </p>
            </div>

            <form action="options.php" method="post">
                <?php
                settings_fields('inomarker_options_group');
                do_settings_sections('inomarker-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * ПРИВАТНЫЙ МЕТОД ДЛЯ СИНХРОНИЗАЦИИ
     *
     * Выполняет запрос к Laravel API, отправляя КЛЮЧ и ДОМЕН,
     * и обновляет статус подписки в базе данных WordPress.
     *
     * @param string $api_key API-ключ для проверки.
     * @return bool True в случае успеха, False в случае ошибки.
     */
    private function sync_subscription_status($api_key) {
        if(empty($api_key)) {
            delete_option('inomarker_subscription_status');
            delete_option('inomarker_last_sync_time');
            return false;
        }

        $site_url = get_site_url();
        $client_domain = wp_parse_url($site_url, PHP_URL_HOST);
        $client_domain = str_replace('www.', '', $client_domain);

        if(!$client_domain) {
//            error_log('Inomarker Plugin Error: Домен не определен';
            delete_option('inomarker_subscription_status');
            return false;
        }

        $response = wp_remote_post('https://inomarker.ru/api/v1/plugin/validate', [
            'method'  => 'POST',
            'timeout' => 15,
            'body'    => [
                'api_key' => $api_key,
                'domain'  => $client_domain,
            ],
        ]);

        if(is_wp_error($response)) {
//            error_log('Inomarker Plugin Error: ' . $response->get_error_message());
            delete_option('inomarker_subscription_status');
            return false;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if($response_code >= 400 || ($data['status'] ?? 'inactive') !== 'active') {
            $error_message = $data['message'] ?? 'Ключ недействителен или подписка истекла.';
            update_option('inomarker_subscription_status', [
                'status'  => 'inactive',
                'message' => $error_message
            ]);
            delete_option('inomarker_last_sync_time');
            return false;
        }

        if($response_code === 200 && ($data['status'] ?? '') === 'active') {
            update_option('inomarker_subscription_status', $data);
            update_option('inomarker_last_sync_time', time());

            $this->fetch_regex_data($api_key);

            return true;
        }

        delete_option('inomarker_subscription_status');
        return false;
    }

    /**
     * ХУК ВАЛИДАЦИИ (Sanitize Callback)
     * Вызывается при сохранении поля 'inomarker_api_key'.
     *
     * @param string $api_key Ключ, который ввел пользователь.
     * @return string Очищенный ключ для сохранения в базу.
     */
    public function validate_and_save_api_key($api_key) {
        $sanitized_key = sanitize_text_field($api_key);

        $this->clear_all_inomarker_caches();

        $is_valid = $this->sync_subscription_status($sanitized_key);

        if($is_valid) {
            add_settings_error(
                'inomarker_messages',
                'api_key_success',
                'Настройки сохранены',
                'success'
            );
        } else {
            $statusData = get_option('inomarker_subscription_status', []);
            $error_message = $statusData['message'] ?? 'Неизвестная ошибка. Пожалуйста, проверьте ключ';

            add_settings_error(
                'inomarker_messages',
                'api_key_error',
                'Ошибка: ' . esc_html($error_message),
                'error'
            );
        }

        return $sanitized_key;
    }

    /**
     * ФОНОВАЯ ЗАДАЧА (WP-CRON)
     * * Этот метод запускается автоматически по расписанию (раз в сутки).
     * Он проверяет статус подписки и скачивает свежие данные.
     */
    public function run_daily_sync() {
        $api_key = get_option('inomarker_api_key');
        if(empty($api_key)) {
            return;
        }

        $is_valid = $this->sync_subscription_status($api_key);

        if($is_valid) {
            $this->fetch_regex_data($api_key);
        } else {
            update_option('inomarker_enabled', false);
        }
    }

    /**
     * Обертка для "тяжелой" функции. Сначала проверяет кэш.
     *
     * @param string $content HTML-контент поста.
     * @return string Модифицированный HTML-контент.
     */
    public function mark_text_in_content($content) {
        if(is_admin() || is_feed() || !in_the_loop() || !is_main_query()) {
            return $content;
        }

        $post_id = get_the_ID();
        if(!$post_id) {
            return $content;
        }
        $cache_key = 'inomarker_cache_pid_' . $post_id;

        $cached_content = get_transient($cache_key);
        if(false !== $cached_content) {
            return $cached_content;
        }

        $processed_content = $this->perform_marking_logic($content);

        set_transient($cache_key, $processed_content, HOUR_IN_SECONDS);

        return $processed_content;
    }

    /**
     * Главная функция, которая "на лету" маркирует текст статей.
     *
     * @param string $content HTML-контент поста.
     * @return string Модифицированный HTML-контент.
     */
    public function perform_marking_logic($content) {

        if(!get_option('inomarker_enabled', false)) {
            return $content;
        }

        $statusData = get_option('inomarker_subscription_status', ['status' => 'unknown']);
        if(($statusData['status'] ?? 'unknown') !== 'active') {
            return $content;
        }

        $mark_style = get_option('inomarker_mark_style', 'inline');

        if(!is_singular()) {
            $mark_style = 'inline';
        }

        $uploads = wp_upload_dir();
        $file_path = $uploads['basedir'] . '/inomarker-data.json';
        if(!file_exists($file_path)) return $content;
        $json_data = file_get_contents($file_path);
        $all_entities = json_decode($json_data, true);
        if(!$all_entities || empty($all_entities)) return $content;

        $all_regex_data = $this->prepare_regex_data($all_entities);
        if(empty($all_regex_data)) return $content;

        $this->footnotes_buffer = [];
        $this->log_buffer = [];
        $dom = new \DOMDocument();
        $html_prefix = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>';
        $html_suffix = '</body></html>';
        @$dom->loadHTML($html_prefix . $content . $html_suffix, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

        $xpath = new \DOMXPath($dom);
        $textNodes = $xpath->query('//text()[not(ancestor::script) and not(ancestor::style) and not(ancestor::mark) and not(ancestor::a) and not(ancestor::h1) and not(ancestor::h2) and not(ancestor::h3) and not(ancestor::pre) and not(ancestor::code)]');

        foreach ($textNodes as $node) {
            $this->process_text_node($node, $all_regex_data, $mark_style);
        }

        if(empty($this->log_buffer)) {
            return $content;
        }

        $body_node = $dom->getElementsByTagName('body')->item(0);
        $final_html = '';
        if($body_node) {
            foreach ($body_node->childNodes as $child) {
                $final_html .= $dom->saveHTML($child);
            }
        } else {
            $final_html = $dom->saveHTML();
        }

        $final_html = html_entity_decode($final_html, ENT_QUOTES, 'UTF-8');

        if($mark_style === 'footnote' && !empty($this->footnotes_buffer)) {
            $final_html .= $this->generate_footnote_html();
        }

        if(get_option('inomarker_show_watermark', true)) {
            if(!empty($this->log_buffer)) {
                $final_html .= '<div class="inomarker-watermark">';
                $final_html .= 'Маркировка произведена сервисом ИноМаркер';
                $final_html .= '</div>';
            }
        }

        $this->footnotes_buffer = [];
        return $final_html;
    }

    /**
     * Готовит массив с регулярными выражениями для обработки.
     * (Исправлено: убрана зависимость от config())
     */
    private function prepare_regex_data($all_entities) {
        $data = [];
        $config = [
            'foreign_agent' => [
                'disclaimer' => '${name} (лицо, выполняющее функции иностранного агента)',
                'label'      => 'лицо, выполняющее функции иностранного агента',
                'marker'     => '*'
            ],
            'terrorist'     => [
                'disclaimer' => '${name} (террористическая организация, запрещенная в РФ)',
                'label'      => 'террористическая организация, запрещенная в РФ',
                'marker'     => '**'
            ],
            'extremist'     => [
                'disclaimer' => '${name} (экстремистская организация, запрещенная в РФ)',
                'label'      => 'экстремистская организация, запрещенная в РФ',
                'marker'     => '***'
            ],
        ];

        foreach ($all_entities as $type => $entities) {
            if(!isset($config[$type])) continue;

            foreach ($entities as $name => $regexes) {
                foreach ([
                             'full',
                             'short'
                         ] as $key) {
                    $pattern = $regexes[$key] ?? null;
                    if(empty($pattern)) continue;

                    $data[] = [
                        'pattern'    => $pattern,
                        'type'       => $type,
                        'name'       => $name,
                        'disclaimer' => str_replace('${name}', $name, $config[$type]['disclaimer']),
                        'label'      => $config[$type]['label'],
                        'marker'     => $config[$type]['marker'],
                    ];
                }
            }
        }
        return $data;
    }

    /**
     * "УМНЫЙ" МЕТОД ОБРАБОТКИ УЗЛА
     * * Находит все совпадения, фильтрует пересечения и безопасно заменяет текст.
     */
    private function process_text_node(\DOMText $node, $all_regex_data, $mark_style) {
        $text = $node->nodeValue;
        $all_matches = [];

        foreach ($all_regex_data as $data) {
            try {
                @preg_match_all('~' . $data['pattern'] . '~iu', $text, $matches, PREG_OFFSET_CAPTURE);

                if(empty($matches[0])) continue;

                foreach ($matches[0] as $match_data) {
                    $all_matches[] = [
                        'start' => $match_data[1],
                        'end'   => $match_data[1] + strlen($match_data[0]),
                        'text'  => $match_data[0],
                        'data'  => $data,
                    ];
                }
            } catch (\Exception $e) {
//                error_log('Inomarker Plugin Error (Regex Failed): ' . $data['pattern']);
            }
        }

        if(empty($all_matches)) return;

        usort($all_matches, function ($a, $b) {
            if($a['start'] === $b['start']) {
                return ($b['end'] - $b['start']) <=> ($a['end'] - $a['start']);
            }
            return $a['start'] <=> $b['start'];
        });

        $final_matches = [];
        $last_end_pos = -1;

        foreach ($all_matches as $match) {
            if($match['start'] >= $last_end_pos) {
                $following_text = substr($text, $match['end'], 20);
                if(preg_match('/^\s*(\*|\()/', $following_text)) {
                    continue;
                }
                $final_matches[] = $match;
                $last_end_pos = $match['end'];
            }
        }

        if(empty($final_matches)) return;

        $dom = $node->ownerDocument;
        $fragment = $dom->createDocumentFragment();
        $last_offset = 0;

        foreach ($final_matches as $match) {
            $fragment->appendChild($dom->createTextNode(substr($text, $last_offset, $match['start'] - $last_offset)));

            $match_text = $match['text'];
            $match_data = $match['data'];

            $this->log_buffer[] = [
                'name' => $match_data['name'],
                'type' => $match_data['type'],
            ];

            if($mark_style === 'footnote') {
                $marker = $match_data['marker'];
                $fragment->appendChild($dom->createTextNode($match_text));
                $sup = $dom->createElement('sup', $marker);
                $fragment->appendChild($sup);
                $this->footnotes_buffer[$marker]['label'] = $match_data['label'];
                $this->footnotes_buffer[$marker]['names'][$match_data['name']] = true;
            } else {
                $mark = $dom->createElement('span', $match_text);
                $mark->setAttribute('class', 'inomarker-highlight');
                $fragment->appendChild($mark);
                $fragment->appendChild($dom->createTextNode(' (' . esc_html($match_data['label']) . ')'));
            }

            $last_offset = $match['end'];
        }

        $fragment->appendChild($dom->createTextNode(substr($text, $last_offset)));
        $node->parentNode->replaceChild($fragment, $node);
    }

    /**
     * Вспомогательный метод для генерации HTML-блока сносок.
     * (Исправленная версия, которая не использует config())
     */
    private function generate_footnote_html() {
        $html = '<div class="inomarker-footnotes">';

        $labels = [];
        $names_by_marker = [];

        foreach ($this->footnotes_buffer as $marker => $data) {
            $labels[$marker] = $data['label'];
            $names_by_marker[$marker] = implode(', ', array_keys($data['names']));
        }

        ksort($names_by_marker);

        foreach ($names_by_marker as $marker => $names_list) {
            $label = $labels[$marker] ?? 'статус не определен.';
            $html .= '<p>' . esc_html($marker . ' ' . $names_list) . ' — ' . esc_html($label) . '</p>';
        }

        $html .= '</div>';
        return $html;
    }

    /**
     * Очищает кэш для конкретного поста при его сохранении.
     */
    public function clear_post_cache_on_save( $post_id ) {
        if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $cache_key = 'inomarker_cache_pid_' . $post_id;
        delete_transient( $cache_key );
    }

    /**
     * Очищает ВЕСЬ кэш плагина.
     * Используется, когда меняются глобальные настройки.
     */
    private function clear_all_inomarker_caches() {
        global $wpdb;

        $transient_keys = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->prefix}options WHERE option_name LIKE %s",
                '\_transient\_inomarker\_cache\_pid\_%'
            )
        );

        $timeout_keys = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->prefix}options WHERE option_name LIKE %s",
                '\_transient\_timeout\_inomarker\_cache\_pid\_%'
            )
        );

        $all_keys_to_delete = array_merge($transient_keys, $timeout_keys);

        if(!empty($all_keys_to_delete)) {
            foreach ($all_keys_to_delete as $key) {
                delete_option($key);
            }
        }
    }

    /**
     * НОВЫЕ МЕТОДЫ: "Прокладки" для `register_settings`.
     * Они очищают кэш и возвращают чистое значение.
     */
    public function sanitize_bool_and_clear_cache( $value ) {
        $this->clear_all_inomarker_caches();
        return (bool) $value;
    }

    public function sanitize_string_and_clear_cache( $value ) {
        $this->clear_all_inomarker_caches();
        return sanitize_text_field( $value );
    }

    /**
     * Выполняется при ДЕАКТИВАЦИИ плагина.
     * Наша задача - "почистить" за собой и убрать задачу из планировщика.
     */
    public static function on_deactivation() {
        wp_clear_scheduled_hook('inomarker_daily_sync');
    }

    /**
     * Обеспечивает, что у нас есть только один экземпляр класса (Singleton).
     */
    public static function instance() {
        if(is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * ПОКАЗ УВЕДОМЛЕНИЙ В АДМИНКЕ
     *
     * Показывает уведомления на всех страницах админ-панели,
     * если подписка истекла ИЛИ скоро истечет.
     */
    public function show_admin_notices() {
        $statusData = get_option('inomarker_subscription_status');
        $is_plugin_enabled_by_user = get_option('inomarker_enabled', false);

        if(!$is_plugin_enabled_by_user) {
            return;
        }

        $status = $statusData['status'] ?? 'unknown';
        $settings_url = admin_url('options-general.php?page=inomarker-settings');
        $pricing_url = 'https://inomarker.ru/pricing';

        if($status !== 'active') {
            $message = $statusData['message'] ?? 'Ваша подписка Inomarker неактивна.';

            echo '<div class="notice notice-error is-dismissible inomarker-notice">';
            echo '<p><strong>[Inomarker] Маркировка на сайте ОТКЛЮЧЕНА</strong> ' . esc_html($message) . '</p>';
            echo '<p><a href="' . esc_url($settings_url) . '" class="button button-secondary">Проверить настройки</a> <a href="' . esc_url($pricing_url) . '" target="_blank" class="button button-primary">Продлить подписку</a></p>';
            echo '</div>';
            return;
        }

        if(isset($statusData['expires_at'])) {

            $expires_timestamp = strtotime($statusData['expires_at']);
            $now_timestamp = time();

            $days_remaining = ceil(($expires_timestamp - $now_timestamp) / 86400);

            if($days_remaining > 0 && $days_remaining <= 3) {

                if($days_remaining == 1) {
                    $days_text = '1 день';
                } else {
                    $days_text = "{$days_remaining} дня";
                }
                $message = 'Ваша подписка Inomarker истекает. Осталось: ' . $days_text;

                echo '<div class="notice notice-warning is-dismissible inomarker-notice">';
                echo '<p><strong>[Inomarker] Внимание!</strong> ' . esc_html($message) . '<br>Чтобы избежать отключения маркировки, пожалуйста, продлите подписку</p>';
                echo '<p><a href="' . esc_url($pricing_url) . '" target="_blank" class="button button-primary">Продлить сейчас</a></p>';
                echo '</div>';
            }
        }
    }

}

/**
 * Вспомогательная функция для добавления кастомного интервала WP-Cron (для теста).
 */
function inomarker_add_cron_intervals($schedules) {
    $schedules['every_five_minutes'] = array(
        'interval' => 300,
        // 5 минут в секундах
        'display'  => 'Every Five Minutes',
    );
    return $schedules;
}

add_filter('cron_schedules', 'inomarker_add_cron_intervals');


/**
 * Глобальная функция-загрузчик.
 * @return Inomarker_Plugin
 */
function Inomarker() {
    return Inomarker_Plugin::instance();
}

// -------------------------------------------------
// ЗАПУСК ПЛАГИНА
// -------------------------------------------------

// Регистрируем наши хуки активации и деактивации
register_activation_hook(__FILE__, [
    'Inomarker_Plugin',
    'on_activation'
]);
register_deactivation_hook(__FILE__, [
    'Inomarker_Plugin',
    'on_deactivation'
]);

// Запускаем плагин
Inomarker();
