<?php
/**
 * Plugin Name:       Иномаркер
 * Plugin URI:        https://inomarker.ru/widgets/wordpress
 * Description:       Автоматически помечает упоминания иностранных агентов, террористов и экстремистов в тексте ваших статей.
 * Version:           1.3.1
 * Author:            Inomarker Team
 * Author URI:        https://inomarker.ru
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       inomarker
 *
 * @package           Inomarker
 */

// Защита от прямого доступа к файлам
if(!defined('ABSPATH')) {
    exit;
}

/**
 * Основной класс плагина, который инициализирует всю логику.
 * Мы используем шаблон Singleton, чтобы гарантировать только один экземпляр класса.
 */
final class Inomarker_Plugin
{

    /**
     * Версия плагина.
     *
     * @var string
     */
    public $version = '1.3.1';

    /**
     * Единственный экземпляр класса (Singleton).
     *
     * @var Inomarker_Plugin|null
     */
    private static $_instance = null;

    /**
     * Хранение сносок в течение одной обработки.
     * @var array
     */
    private $footnotes_buffer = [];

    private $log_buffer = [];

    private $stats_page_hook = '';

    public function __construct() {

        // Регистрируем страницу в меню админ-панели
        add_action('admin_menu', [
            $this,
            'add_admin_menu'
        ]);

        // Регистрация настроек (полей)
        add_action('admin_init', [
            $this,
            'register_settings'
        ]);

        // Включение стилей для админ-панели (для красивого статуса)
        add_action('admin_enqueue_scripts', [
            $this,
            'enqueue_admin_styles'
        ]);

        // Это связывает нашу фоновую задачу (WP-Cron) с методом 'run_daily_sync'
        add_action('inomarker_daily_sync', [
            $this,
            'run_daily_sync'
        ]);

        add_filter('the_content', [
            $this,
            'mark_text_in_content'
        ],         10);

        add_action('admin_notices', [
            $this,
            'show_admin_notices'
        ]);

        add_action('save_post', [
            $this,
            'clear_post_cache_on_save'
        ]);

        add_action('rest_api_init', [
            $this,
            'register_rest_routes'
        ]);

        add_action('wp_footer', [
            $this,
            'print_footer_scripts'
        ]);

        add_action('wp_ajax_inomarker_toggle_exclusion', [
            $this,
            'ajax_toggle_exclusion'
        ]);

        add_action('wp_ajax_inomarker_delete_stat', [
            $this,
            'ajax_delete_stat'
        ]);

        // Проверка наличия обновления (запрос к списку обновлений)
        add_filter('site_transient_update_plugins', [
            $this,
            'check_for_update'
        ]);

        // Вывод информации о плагине в модальном окне (Details)
        add_filter('plugins_api', [
            $this,
            'plugin_popup_info'
        ],         20, 3);

    }

    /**
     * Добавляет пункты "Inomarker" в меню администратора.
     */
    public function add_admin_menu() {

        add_menu_page(
            'Иномаркер', // Заголовок страницы
            'Иномаркер', // Название меню
            'manage_options',
            'inomarker-settings',
            [ $this, 'render_settings_page' ],
            'dashicons-shield',
            80
        );

        // Подменю настроек
        add_submenu_page(
            'inomarker-settings',
            'Настройки',
            'Настройки',
            'manage_options',
            'inomarker-settings',
            [ $this, 'render_settings_page' ]
        );

        // Подменю статистики
        $this->stats_page_hook = add_submenu_page(
            'inomarker-settings',
            'Статистика',
            'Статистика',
            'manage_options',
            'inomarker-stats',
            [ $this, 'render_stats_page' ]
        );

    }

    /**
     * Регистрирует параметры в базе данных и создает разделы настроек.
     */
    public function register_settings() {
        // 1. Ключ API
        register_setting('inomarker_options_group', 'inomarker_api_key', [
            'type'              => 'string',
            'sanitize_callback' => [
                $this,
                'validate_and_save_api_key'
            ],
            'default'           => ''
        ]);

        // 2. Переключатель "Вкл."/Выключен"
        register_setting('inomarker_options_group', 'inomarker_enabled', [
            'type'              => 'boolean',
            'sanitize_callback' => [
                $this,
                'sanitize_bool_and_clear_cache'
            ],
            'default'           => false
        ]);

        // 3. Стиль маркировки
        register_setting('inomarker_options_group', 'inomarker_mark_style', [
            'type'              => 'string',
            'sanitize_callback' => [
                $this,
                'sanitize_string_and_clear_cache'
            ],
            'default'           => 'inline'
        ]);

        // 4. "Водяной знак"
        register_setting('inomarker_options_group', 'inomarker_show_watermark', [
            'type'              => 'boolean',
            'sanitize_callback' => [
                $this,
                'sanitize_bool_and_clear_cache'
            ],
            'default'           => true,
        ]);

        // --- РАЗДЕЛ 1: Подключение ---
        add_settings_section(
            'inomarker_connection_section',
            'Подключение к сервису',
            null,
            'inomarker-settings'
        );

        add_settings_field(
            'inomarker_api_key',
            'API Ключ Виджета',
            [
                $this,
                'render_api_key_field'
            ],
            'inomarker-settings',
            'inomarker_connection_section'
        );

        // --- РАЗДЕЛ 2. Настройки дисплея ---
        add_settings_section(
            'inomarker_display_section',
            'Настройки маркировки',
            null,
            'inomarker-settings'
        );

        add_settings_field(
            'inomarker_enabled',
            'Включить автомаркировку',
            [
                $this,
                'render_enabled_field'
            ],
            'inomarker-settings',
            'inomarker_display_section'
        );

        add_settings_field(
            'inomarker_mark_style',
            'Стиль маркировки',
            [
                $this,
                'render_style_field'
            ],
            'inomarker-settings',
            'inomarker_display_section'
        );

        add_settings_field(
            'inomarker_show_watermark',
            'Водный знак',
            [
                $this,
                'render_watermark_field'
            ],
            'inomarker-settings',
            'inomarker_display_section'
        );

    }

    /**
     * HTML для поля ключа API.
     */
    public function render_api_key_field() {
        $value = get_option( 'inomarker_api_key' );
        echo '<input type="text" name="inomarker_api_key" value="' . esc_attr( $value ) . '" class="regular-text code">';
        echo '<p class="description">Скопируйте ключ из вашего ';
        echo '<a href="https://inomarker.ru/widgets" target="_blank">личного кабинета</a>';
        echo '</p>';
    }

    /**
     * HTML для флажка включения.
     */
    public function render_enabled_field() {
        $value = get_option('inomarker_enabled');
        $statusData = get_option('inomarker_subscription_status');

        $status = $statusData['status'] ?? 'unknown';
        $isDisabled = ($status !== 'active');

        ?>
        <label class="switch">
            <input type="checkbox" name="inomarker_enabled" value="1" <?php checked(1, $value); ?> <?php disabled($isDisabled); ?>>
            <span class="slider round"></span>
        </label>
        <?php
        if ( $isDisabled ) {
            echo '<p class="description" style="color: #d63638;">Требуется активная подписка для включения</p>';
        }
    }

    /**
     * HTML для выбора стиля.
     */
    public function render_style_field() {
        $value = get_option( 'inomarker_mark_style', 'inline' );
        ?>
        <select name="inomarker_mark_style">
            <option value="inline" <?php selected( $value, 'inline' ); ?>>В тексте (в скобках)</option>
            <option value="footnote" <?php selected( $value, 'footnote' ); ?>>Сносками (*)</option>
        </select>
        <p class="description">
            <strong>В тексте:</strong> Иван Иванов (иноагент)<br>
            <strong>Сносками:</strong> Иван Иванов*. (Сноска будет добавлена в конец статьи)
        </p>
        <?php
    }

    /**
     * HTML для установки флажка "Водяной знак".
     */
    public function render_watermark_field() {
        $value = get_option('inomarker_show_watermark', true);

        echo '<label class="switch">';
        echo '<input type="checkbox" name="inomarker_show_watermark" value="1" ' . checked(1, $value, false) . '>';
        echo '<span class="slider round"></span>';
        echo '</label>';
        echo '<p class="description">Добавляет в конец статьи строку "Маркировка произведена сервисом ИноМаркер"</p>';
    }

    /**
     * Это выполняется при активации плагина.
     * Наша главная задача здесь - запланировать ежедневное фоновое задание.
     */
    public static function on_activation() {
        if(!wp_next_scheduled('inomarker_daily_sync')) {
            wp_schedule_event(time(), 'daily', 'inomarker_daily_sync');
        }

        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // Таблица для статистики страниц
        $table_stats = $wpdb->prefix . 'inomarker_stats';

        $sql = "CREATE TABLE $table_stats (
            id BIGINT(20) NOT NULL AUTO_INCREMENT,
            url_hash VARCHAR(32) NOT NULL,
            url TEXT NOT NULL,
            found_data LONGTEXT NULL, -- JSON with found persons and types
            excluded_data LONGTEXT NULL, -- JSON with exceptions for this page
            has_short TINYINT(1) DEFAULT 0,
            last_updated DATETIME DEFAULT '0000-00-00 00:00:00',
            PRIMARY KEY  (id),
            UNIQUE KEY url_hash (url_hash)
        ) $charset_collate;";

        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );

    }

    /**
     * ЗАГРУЗКА НАСТРОЕК
     * * Вызывается из 'run_daily_sync', если подписка активна.
     * Загружает файл data.json.
     *
     * @param string $api_key API-ключ для аутентификации.
     */
    private function fetch_regex_data($api_key) {
        // 1. Создание URL-адреса для конечной точки
        $url = 'https://inomarker.ru/api/v1/plugin/regex-data';
        $request_url = add_query_arg('api_key', $api_key, $url);

        // 2. Отправка запроса GET
        $response = wp_remote_get($request_url, [
            'timeout' => 30,
        ]);

        // 3. Обработка ошибок
        if(is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            $error_message = is_wp_error($response) ? $response->get_error_message() : 'Не удалось получить данные с сервера';
            error_log('Ошибка плагина Inomarker (fetch_regex_data): ' . $error_message);
            return;
        }

        // 4. Мы получаем текст ответа (это наш JSON-файл в виде строки)
        $json_data = wp_remote_retrieve_body($response);

        // 5. Сохранение файла JSON в папку /wp-content/uploads/
        $uploads = wp_upload_dir();
        $file_path = $uploads['basedir'] . '/inomarker-data.json';

        if(file_put_contents($file_path, $json_data) === false) {
            error_log('Ошибка плагина Inomarker: Не удалось записать data.json в ' . $file_path);
        } else {
            // Мы записываем время успешного обновления, чтобы отобразить его в админке
            update_option('inomarker_last_data_sync_time', time());
        }
    }

    /**
     * Включает CSS для панели администратора.
     */
    public function enqueue_admin_styles( $hook ) {
        wp_register_style( 'inomarker-admin-style', false );
        $css = '
        .inomarker-status-card {background: #fff; border: 1px solid #c3c4c7; padding: 20px; border-radius: 5px; margin-bottom: 20px; border-left: 4px solid #72aee6;}
        .inomarker-status-active { border-left-color: #00a32a; }
        .inomarker-status-inactive { border-left-color: #d63638; }
        .im-badge { padding: 5px 10px; border-radius: 4px; color: #fff; font-weight: bold; font-size: 12px; }
        .im-badge-green { background: #00a32a; }
        .im-badge-red { background: #d63638; }
        .im-badge-gray { background: #646970; }
        .switch { position: relative; display: inline-block; width: 60px; height: 34px; }
        .switch input { opacity: 0; width: 0; height: 0; }
        .slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: #ccc; transition: .4s; }
        .slider:before { position: absolute; content: ""; height: 26px; width: 26px; left: 4px; bottom: 4px; background-color: white; transition: .4s; }
        input:checked + .slider { background-color: #2196F3; }
        input:focus + .slider { box-shadow: 0 0 1px #2196F3; }
        input:checked + .slider:before { transform: translateX(26px); }
        .slider.round { border-radius: 34px; }
        .slider.round:before { border-radius: 50%; }
        .inomarker-notice p { padding-bottom: 5px; }
        .inomarker-notice .button-primary { font-weight: bold; }
        .im-tag {display: inline-flex; align-items: center; gap: 4px;background: #e0e7ff; color: #3730a3; border: 1px solid #c7d2fe;padding: 2px 8px; border-radius: 12px; font-size: 12px; cursor: pointer; margin: 2px;transition: all 0.2s;}
        .im-tag:hover { opacity: 0.8; }
        .im-tag-short { border-color: #fcd34d; background: #fffbeb; color: #92400e; }
        .im-tag-excluded {background: #f3f4f6; color: #9ca3af; border-color: #e5e7eb; text-decoration: line-through;}
        .im-tag-excluded:hover { text-decoration: none; background: #fee2e2; color: #991b1b; border-color: #fca5a5; }
        .button-link-delete {background: none;border: none;color: #a0a5b9;cursor: pointer;padding: 5px;transition: color 0.2s;}
        .button-link-delete:hover {color: #d63638;}
        .button-link-delete .dashicons {font-size: 18px;width: 18px;height: 18px;}
        ';

        wp_add_inline_style( 'inomarker-admin-style', $css );

        wp_enqueue_style( 'inomarker-admin-style' );

        if ( $hook === $this->stats_page_hook ) {

            wp_register_script( 'inomarker-admin-js', false, ['jquery'], '1.0', true );

            $js_vars = [
                'nonce_exclude'   => esc_js(wp_create_nonce('inomarker_exclude')),
                'nonce_delete'    => esc_js(wp_create_nonce('inomarker_delete')),
                'error_msg'       => 'Ошибка сохранения настроек',
                'delete_msg_sure' => 'Вы уверены, что хотите удалить статистику для этой страницы?',
                'ajax_url'        => admin_url('admin-ajax.php')
            ];

            wp_add_inline_script( 'inomarker-admin-js', 'const InomarkerParams = ' . wp_json_encode( $js_vars ) . ';' );

            $js_code = "
            function toggleExclusion(btn, id, name) {
                var data = {
                    action: 'inomarker_toggle_exclusion',
                    id: id,
                    name: name,
                    _ajax_nonce: InomarkerParams.nonce_exclude
                };

                btn.style.opacity = '0.5';

                jQuery.post(InomarkerParams.ajax_url, data, function(response) {
                    btn.style.opacity = '1';
                    if (response.success) {
                        if (response.data.status === 'excluded') {
                            btn.classList.remove('im-tag-active');
                            btn.classList.add('im-tag-excluded');
                        } else {
                            btn.classList.remove('im-tag-excluded');
                            btn.classList.add('im-tag-active');
                        }
                    } else {
                        alert(InomarkerParams.error_msg);
                    }
                });
            }
            function deletePageStat(btn, id) {
                if (!confirm(InomarkerParams.delete_msg_sure)) {
                    return;
                }

                var row = btn.closest('tr');

                btn.disabled = true;
                row.style.opacity = '0.5';

                var data = {
                    action: 'inomarker_delete_stat',
                    id: id,
                    _ajax_nonce: InomarkerParams.nonce_delete
                };

                jQuery.post(InomarkerParams.ajax_url, data, function(response) {
                    if (response.success) {
                        row.style.backgroundColor = '#ffebee';
                        jQuery(row).fadeOut(400, function() {
                            jQuery(this).remove();
                            if (jQuery('.wp-list-table tbody tr').length === 0) {
                                location.reload();
                            }
                        });
                    } else {
                        alert('Ошибка: ' + response.data);
                        row.style.opacity = '1';
                        btn.disabled = false;
                    }
                });
            }
            ";

            wp_add_inline_script( 'inomarker-admin-js', $js_code );
            wp_enqueue_script( 'inomarker-admin-js' );
        }
    }

    /**
     * Рендеринг всей страницы настроек.
     */
    public function render_settings_page() {
        // Мы получаем данные о статусе (которые мы будем обновлять через Cron).
        $subStatus = get_option('inomarker_subscription_status', ['status' => 'unknown']);
        $status = $subStatus['status'] ?? 'unknown';
        $tariff = $subStatus['tariff_name'] ?? 'Неизвестно';
        $expires = isset($subStatus['expires_at']) ? date_i18n('j F Y', strtotime($subStatus['expires_at'])) : '—';

        $is_plugin_enabled = get_option('inomarker_enabled', false);
        $lastSync = get_option('inomarker_last_sync_time') ? date_i18n('j F Y H:i', get_option('inomarker_last_sync_time')) : 'Никогда';
        $lastDataSync = get_option('inomarker_last_data_sync_time') ? date_i18n('j F Y H:i', get_option('inomarker_last_data_sync_time')) : 'Никогда';

        ?>
        <div class="wrap">
            <h1>Иномаркер</h1>

            <div
                class="inomarker-status-card <?php echo esc_attr( ($status === 'active') ? 'inomarker-status-active' : 'inomarker-status-inactive' ); ?>">
                <h2>Статус подписки</h2>
                <p>
                    <strong>Текущее состояние:</strong>
                    <?php if ( $status === 'active' ) : ?>
                        <span class="im-badge im-badge-green">АКТИВНА</span>
                    <?php elseif ( $status === 'unknown' ) : ?>
                        <span class="im-badge im-badge-gray">НЕ ПРОВЕРЕНО</span>
                    <?php else : ?>
                        <span class="im-badge im-badge-red">НЕАКТИВНА</span>
                    <?php endif; ?>
                </p>
                <p>
                    <strong>Маркировка на сайте:</strong>
                    <?php if ( $is_plugin_enabled && $status === 'active' ) : ?>
                        <span class="im-badge im-badge-green">ВКЛЮЧЕНА</span>
                    <?php else : ?>
                        <span class="im-badge im-badge-red">ВЫКЛЮЧЕНА</span>
                    <?php endif; ?>
                </p>
                <p><strong>Тариф:</strong> <?php echo esc_html( $tariff ); ?></p>
                <p><strong>Истекает:</strong> <?php echo esc_html( $expires ); ?></p>
                <hr>
                <p class="description">
                    Последняя проверка статуса: <?php echo esc_html( $lastDataSync ); ?>
                </p>
            </div>

            <form action="options.php" method="post">
                <?php
                settings_fields('inomarker_options_group');
                do_settings_sections('inomarker-settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * ЧАСТНЫЙ МЕТОД СИНХРОНИЗАЦИИ
     *
     * Выполняет запрос к API, отправляя КЛЮЧ и ДОМЕН,
     * и обновляет статус подписки в базе данных WordPress.
     *
     * @param string $api_key API ключ для проверки.
     * @return bool Истинно в случае успеха, ложно в случае ошибки.
     */
    private function sync_subscription_status($api_key) {
        if(empty($api_key)) {
            delete_option('inomarker_subscription_status');
            delete_option('inomarker_last_sync_time');
            return false;
        }

        $site_url = get_site_url();
        $client_domain = wp_parse_url($site_url, PHP_URL_HOST);
        $client_domain = str_replace('www.', '', $client_domain);

        if(!$client_domain) {
            error_log('Ошибка плагина Inomarker: Домен не определен');
            delete_option('inomarker_subscription_status');
            return false;
        }

        $response = wp_remote_post('https://inomarker.ru/api/v1/plugin/validate', [
            'method'  => 'POST',
            'timeout' => 15,
            'body'    => [
                'api_key' => $api_key,
                'domain'  => $client_domain,
            ],
        ]);

        if(is_wp_error($response)) {
            error_log('Ошибка плагина Inomarker: ' . $response->get_error_message());
            delete_option('inomarker_subscription_status');
            return false;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if($response_code >= 400 || ($data['status'] ?? 'inactive') !== 'active') {
            $error_message = $data['message'] ?? 'Ключ недействителен или подписка истекла';
            update_option('inomarker_subscription_status', [
                'status'  => 'inactive',
                'message' => $error_message
            ]);
            delete_option('inomarker_last_sync_time');
            return false;
        }

        if($response_code === 200 && ($data['status'] ?? '') === 'active') {
            update_option('inomarker_subscription_status', $data);
            update_option('inomarker_last_sync_time', time());

            $this->fetch_regex_data($api_key);

            return true;
        }

        delete_option('inomarker_subscription_status');
        return false;
    }

    /**
     * МЕХАНИЗМ ПРОВЕРКИ (очистка обратного вызова)
     * Вызывается при сохранении поля 'inomarker_api_key'.
     *
     * @param string $api_key Ключ, который ввел пользователь.
     * @return string Очищенный ключ для сохранения в базе данных.
     */
    public function validate_and_save_api_key($api_key) {
        $sanitized_key = sanitize_text_field($api_key);

        $this->clear_all_inomarker_caches();

        $is_valid = $this->sync_subscription_status($sanitized_key);

        if($is_valid) {
            add_settings_error(
                'inomarker_messages',
                'api_key_success',
                'Ключ успешно проверен и сохранен. Ваша подписка активна',
                'success'
            );
        } else {
            $statusData = get_option('inomarker_subscription_status', []);
            $error_message = $statusData['message'] ?? 'Неизвестная ошибка. Пожалуйста, проверьте ключ';

            add_settings_error(
                'inomarker_messages',
                'api_key_error',
                'Ошибка: ' . esc_html( $error_message ),
                'error'
            );
        }

        return $sanitized_key;
    }

    /**
     * ФОНОВАЯ ЗАДАЧА (WP-CRON)
     * * Этот метод выполняется автоматически по расписанию (один раз в день).
     * Он проверяет статус подписки и загружает последние данные.
     */
    public function run_daily_sync() {
        $api_key = get_option('inomarker_api_key');
        if(empty($api_key)) {
            return;
        }

        $is_valid = $this->sync_subscription_status($api_key);

        if($is_valid) {
            $this->fetch_regex_data($api_key);
        } else {
            update_option('inomarker_enabled', false);
        }
    }

    /**
     * Оболочка для "тяжелой" функции. Сначала она проверяет кэш.
     *
     * @param string $content HTML-содержимое сообщения.
     * @return string Измененный HTML-контент.
     */
    public function mark_text_in_content($content) {
        $isDebug = $this->is_debug_active();

//        if(is_admin() || is_feed() || !in_the_loop() || !is_main_query()) {
        if(is_admin() || is_feed()) {
            return $content;
        }

        $post_id = get_the_ID();
        if(!$post_id) {
            return $content;
        }

        $context_suffix = is_singular() ? '_single' : '_list';
        $cache_key = 'inomarker_cache_pid_' . $post_id . $context_suffix;

        if(!$isDebug) {
            $cached_content = get_transient($cache_key);
            if(false !== $cached_content) {
                return $cached_content;
            }
        }

        $processed_content = $this->perform_marking_logic($content);

        if(!$isDebug) {
            set_transient($cache_key, $processed_content, HOUR_IN_SECONDS);
        }

        return $processed_content;
    }

    /**
     * Основная функция, которая помечает текст статей на лету.
     *
     * @param string $content HTML-содержимое сообщения.
     * @return string Измененный HTML-контент.
     */
    public function perform_marking_logic($content) {
        $debugLog = [];
        $isDebug = $this->is_debug_active();

        $log = function($msg, $status = 'info') use (&$debugLog, $isDebug) {
            if ($isDebug) $debugLog[] = ['msg' => $msg, 'status' => $status];
        };

        $log("Начата обработка URL-адреса: " . $this->get_clean_url_hash(null));

        if(!get_option('inomarker_enabled', false)) {
            $log('Плагин выключен в настройках (inomarker_enabled = false)', 'error');
            if ($isDebug) return $content . $this->render_debug_console($debugLog);
            return $content;
        }
        $log('Плагин включен', 'success');

        $statusData = get_option('inomarker_subscription_status', ['status' => 'unknown']);
        if(($statusData['status'] ?? 'unknown') !== 'active') {
            $log('Подписка не активирован, статус: ' . ($statusData['status'] ?? 'unknown'), 'error');
            if ($isDebug) return $content . $this->render_debug_console($debugLog);
            return $content;
        }
        $log('Подписка активна', 'success');

        $mark_style = get_option('inomarker_mark_style', 'inline');
        if(!is_singular()) {
            $mark_style = 'inline';
        }

        global $wpdb;
        $url_hash = $this->get_clean_url_hash( null );
        $table_stats = $wpdb->prefix . 'inomarker_stats';
        $excluded_json = $wpdb->get_var( $wpdb->prepare( "SELECT excluded_data FROM $table_stats WHERE url_hash = %s", $url_hash ) );
        $excluded_names = $excluded_json ? json_decode( $excluded_json, true ) : [];
        $excluded_map = $excluded_names ? array_flip( $excluded_names ) : [];

        if (!empty($excluded_names)) {
            $log('Найдены исключения: ' . implode(", ", $excluded_names), 'warning');
        } else {
            $log('Исключения не найдены', 'success');
        }

        $uploads = wp_upload_dir();
        $file_path = $uploads['basedir'] . '/inomarker-data.json';
        if(!file_exists($file_path)) {
            $log("Фаил не найден: $file_path", 'error');
            if ($isDebug) return $content . $this->render_debug_console($debugLog);
            return $content;
        }

        $json_data = file_get_contents($file_path);
        $all_entities = json_decode($json_data, true);
        if(!$all_entities || empty($all_entities)) {
            $log('Невалидный JSON в файле', 'error');
            if ($isDebug) return $content . $this->render_debug_console($debugLog);
            return $content;
        }
        $log("JSON валидный, проверка регулярок...", 'info');

        $all_regex_data = $this->prepare_regex_data($all_entities);

        $filtered_regex_data = [];
        foreach ($all_regex_data as $regex_item) {
            if ( !isset( $excluded_map[ $regex_item['name'] ] ) ) {
                $filtered_regex_data[] = $regex_item;
            } else {
                $log('Пропущена регулярка для: ' . $regex_item['name'], 'warning');
            }
        }
        $all_regex_data = $filtered_regex_data;
        $log('Общее количество активных шаблонов регулярных выражений для сканирования: ' . count($all_regex_data), 'info');

        $this->footnotes_buffer = [];
        $this->log_buffer = [];

//        $dom = new \DOMDocument();
//        if (function_exists('mb_convert_encoding')) {
//            $content = mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8');
//        }
//        libxml_use_internal_errors(true);
//        @$dom->loadHTML('<!DOCTYPE html><html><body>' . $content . '</body></html>', LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
//        libxml_clear_errors();

        $dom = new \DOMDocument();
        $html_prefix = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>';
        $html_suffix = '</body></html>';
        @$dom->loadHTML($html_prefix . $content . $html_suffix, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

        $xpath = new \DOMXPath($dom);
        $textNodes = $xpath->query('//text()[not(ancestor::script) and not(ancestor::style) and not(ancestor::mark) and not(ancestor::a) and not(ancestor::h1) and not(ancestor::h2) and not(ancestor::h3) and not(ancestor::pre) and not(ancestor::code)]');

        $log( 'Нашел ' . $textNodes->length . ' текста для сканирования', 'info');

        $matches_count = 0;
        foreach ($textNodes as $node) {
            $this->process_text_node($node, $all_regex_data, $mark_style);
        }

        if (!empty($this->log_buffer)) {
            foreach($this->log_buffer as $found) {
                $log('Нашли: ' . $found['name'] . ' (' . $found['type'] . ')', 'success');
                $matches_count++;
            }
        } else {
            $log('Не нашли вхождений в тексте', 'warning');
            return $content;
        }

        $body_node = $dom->getElementsByTagName('body')->item(0);
        $final_html = '';

        if($body_node) {
            foreach ($body_node->childNodes as $child) {
                $final_html .= $dom->saveHTML($child);
            }
        } else {
            $final_html = $content;
        }

        if($mark_style === 'footnote' && !empty($this->footnotes_buffer)) {
            $final_html .= $this->generate_footnote_html();
        }

        if(get_option('inomarker_show_watermark', true) && !empty($this->log_buffer)) {
            $final_html .= '<div class="inomarker-watermark">Маркировка произведена сервисом ИноМаркер</div>';
        }

        $this->footnotes_buffer = [];

        if ($isDebug) {
            $final_html .= $this->render_debug_console($debugLog);
        }

        return $final_html;
    }
    public function perform_marking_logic_old($content) {

        if(!get_option('inomarker_enabled', false)) {
            return $content;
        }

        $statusData = get_option('inomarker_subscription_status', ['status' => 'unknown']);
        if(($statusData['status'] ?? 'unknown') !== 'active') {
            return $content;
        }

        $mark_style = get_option('inomarker_mark_style', 'inline');

        if(!is_singular()) {
            $mark_style = 'inline';
        }


        global $wpdb;
        $url_hash = $this->get_clean_url_hash( null );
        $table_stats = $wpdb->prefix . 'inomarker_stats';

        // We get exceptions from the database (one quick query)
        $excluded_json = $wpdb->get_var( $wpdb->prepare( "SELECT excluded_data FROM $table_stats WHERE url_hash = %s", $url_hash ) );// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $excluded_names = $excluded_json ? json_decode( $excluded_json, true ) : [];
        // We turn them into keys for quick search
        $excluded_map = $excluded_names ? array_flip( $excluded_names ) : [];


        $uploads = wp_upload_dir();
        $file_path = $uploads['basedir'] . '/inomarker-data.json';
        if(!file_exists($file_path)) return $content;
        $json_data = file_get_contents($file_path);
        $all_entities = json_decode($json_data, true);
        if(!$all_entities || empty($all_entities)) return $content;

        $all_regex_data = $this->prepare_regex_data($all_entities);
        if(empty($all_regex_data)) return $content;


        $filtered_regex_data = [];
        foreach ($all_regex_data as $regex_item) {
            if ( !isset( $excluded_map[ $regex_item['name'] ] ) ) {
                $filtered_regex_data[] = $regex_item;
            }
        }
        $all_regex_data = $filtered_regex_data;


        $this->footnotes_buffer = [];
        $this->log_buffer = [];
        $dom = new \DOMDocument();
        $html_prefix = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>';
        $html_suffix = '</body></html>';
        @$dom->loadHTML($html_prefix . $content . $html_suffix, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

        $xpath = new \DOMXPath($dom);
        $textNodes = $xpath->query('//text()[not(ancestor::script) and not(ancestor::style) and not(ancestor::mark) and not(ancestor::a) and not(ancestor::h1) and not(ancestor::h2) and not(ancestor::h3) and not(ancestor::pre) and not(ancestor::code)]');

        foreach ($textNodes as $node) {
            $this->process_text_node($node, $all_regex_data, $mark_style);
        }

        if(empty($this->log_buffer)) {
            return $content;
        }

        $body_node = $dom->getElementsByTagName('body')->item(0);
        $final_html = '';
        if($body_node) {
            foreach ($body_node->childNodes as $child) {
                $final_html .= $dom->saveHTML($child);
            }
        } else {
            $final_html = $dom->saveHTML();
        }

        $final_html = html_entity_decode($final_html, ENT_QUOTES, 'UTF-8');

        if($mark_style === 'footnote' && !empty($this->footnotes_buffer)) {
            $final_html .= $this->generate_footnote_html();
        }

        if(get_option('inomarker_show_watermark', true)) {
            if(!empty($this->log_buffer)) {
                $final_html .= '<div class="inomarker-watermark">';
                $final_html .= 'Маркировка произведена сервисом ИноМаркер';
                $final_html .= '</div>';
            }
        }

        $this->footnotes_buffer = [];
        return $final_html;
    }

    private function render_debug_console($logs) {
        $html = '<div style="
        background: #23282d;
        color: #fff;
        padding: 20px;
        margin: 40px 0;
        border: 2px solid #0073aa;
        border-radius: 5px;
        font-family: monospace;
        font-size: 14px;
        line-height: 1.5;
        z-index: 99999;
        position: relative;">';

        $html .= '<h3 style="color:#fff; margin-top:0;">Консоль отладки плагина Иномаркер</h3>';
        $html .= '<p style="color:#ccc; font-style:italic;">Этот блок виден только вам по секретной ссылке</p>';
        $html .= '<hr style="border-color:#555;">';

        foreach ($logs as $entry) {
            $color = '#fff';
            if ($entry['status'] === 'error') $color = '#ff6b6b';
            if ($entry['status'] === 'warning') $color = '#feca57';
            if ($entry['status'] === 'success') $color = '#1dd1a1';

            $html .= '<div style="margin-bottom: 5px;">';
            $html .= '<span style="color:'. $color .'; font-weight:bold;">[' . strtoupper($entry['status']) . ']</span> ';
            $html .= esc_html($entry['msg']);
            $html .= '</div>';
        }

        $html .= '<hr style="border-color:#555;">';
        $html .= '<div><strong>Серверное время:</strong> ' . date('Y-m-d H:i:s') . '</div>';
        $html .= '<div><strong>Версия PHP:</strong> ' . phpversion() . '</div>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Подготавливает массив с регулярными выражениями для обработки.
     */
    private function prepare_regex_data($all_entities) {
        $data = [];
        $config = [
            'foreign_agent' => [
                'disclaimer' => '${name} (лицо, выполняющее функции иностранного агента)',
                'label'      => 'лицо, выполняющее функции иностранного агента',
                'marker'     => '*'
            ],
            'terrorist'     => [
                'disclaimer' => '${name} (террористическая организация, запрещенная в РФ)',
                'label'      => 'террористическая организация, запрещенная в РФ',
                'marker'     => '**'
            ],
            'extremist'     => [
                'disclaimer' => '${name} (экстремистская организация, запрещенная в РФ)',
                'label'      => 'экстремистская организация, запрещенная в РФ',
                'marker'     => '***'
            ],
        ];

        foreach ($all_entities as $type => $entities) {
            if(!isset($config[$type])) continue;

            foreach ($entities as $name => $regexes) {
                foreach ([
                             'full',
                             'short'
                         ] as $key) {
                    $pattern = $regexes[$key] ?? null;
                    if(empty($pattern)) continue;

                    $is_short = ($key === 'short');
                    $base_label = $config[$type]['label'];

                    $prefix = 'Возможно ';

                    $final_label = $is_short ? $prefix . ' ' . $base_label : $base_label;

                    $base_disclaimer = str_replace( '${name}', $name, $config[$type]['disclaimer'] );
                    $final_disclaimer = $is_short ? $prefix . ' ' . $base_disclaimer : $base_disclaimer;

                    $data[] = [
                        'pattern'    => $pattern,
                        'type'       => $type,
                        'name'       => $name,
                        'disclaimer' => $final_disclaimer,
                        'label'      => $final_label,
                        'marker'     => $config[$type]['marker'],
                        'is_short'   => $is_short,
                    ];
                }
            }
        }
        return $data;
    }

    /**
     * СПОСОБ ОБРАБОТКИ "УМНЫХ" УЗЛОВ
     * * Находит все совпадения, фильтрует пересечения и безопасно заменяет текст.
     */
    private function process_text_node(\DOMText $node, $all_regex_data, $mark_style) {
        $text = $node->nodeValue;
        $all_matches = [];

        foreach ($all_regex_data as $data) {
            try {
                @preg_match_all('~' . $data['pattern'] . '~iu', $text, $matches, PREG_OFFSET_CAPTURE);

                if(empty($matches[0])) continue;

                foreach ($matches[0] as $match_data) {
                    $all_matches[] = [
                        'start' => $match_data[1],
                        'end'   => $match_data[1] + strlen($match_data[0]),
                        'text'  => $match_data[0],
                        'data'  => $data,
                    ];
                }
            } catch (\Exception $e) {
                error_log('Ошибка плагина Inomarker (Regex Failed): ' . $data['pattern']);
            }
        }

        if(empty($all_matches)) return;

        usort($all_matches, function ($a, $b) {
            if($a['start'] === $b['start']) {
                return ($b['end'] - $b['start']) <=> ($a['end'] - $a['start']);
            }
            return $a['start'] <=> $b['start'];
        });

        $final_matches = [];
        $last_end_pos = -1;

        foreach ($all_matches as $match) {
            if($match['start'] >= $last_end_pos) {
                $following_text = substr($text, $match['end'], 20);
                if(preg_match('/^\s*(\*|\()/', $following_text)) {
                    continue;
                }
                $final_matches[] = $match;
                $last_end_pos = $match['end'];
            }
        }

        if(empty($final_matches)) return;

        $dom = $node->ownerDocument;
        $fragment = $dom->createDocumentFragment();
        $last_offset = 0;

        foreach ($final_matches as $match) {
            $fragment->appendChild($dom->createTextNode(substr($text, $last_offset, $match['start'] - $last_offset)));

            $match_text = $match['text'];
            $match_data = $match['data'];

            $this->log_buffer[] = [
                'name'     => $match_data['name'],
                'type'     => $match_data['type'],
                'is_short' => $match_data['is_short'],
            ];

            if($mark_style === 'footnote') {
                $marker = $match_data['marker'];
                $fragment->appendChild($dom->createTextNode($match_text));
                $sup = $dom->createElement('sup', $marker);
                $sup->setAttribute( 'class', 'inomarker-footnote-ref' );
                if ($match_data['is_short']) {
                    $sup->setAttribute( 'class', 'inomarker-footnote-ref inomarker-possible' );
                }
                $fragment->appendChild($sup);
                $this->footnotes_buffer[$marker]['label'] = $match_data['label'];
                $this->footnotes_buffer[$marker]['names'][$match_data['name']] = true;
            } else {
                $mark = $dom->createElement('span', $match_text);
                $mark->setAttribute('class', 'inomarker-highlight');
                $fragment->appendChild($mark);
                $fragment->appendChild($dom->createTextNode(' (' . esc_html($match_data['label']) . ')'));
            }

            $last_offset = $match['end'];
        }

        $fragment->appendChild($dom->createTextNode(substr($text, $last_offset)));
        $node->parentNode->replaceChild($fragment, $node);
    }

    /**
     * Вспомогательный метод для генерации HTML-блока сносок.
     */
    private function generate_footnote_html() {
        $html = '<div class="inomarker-footnotes">';

        $labels = [];
        $names_by_marker = [];

        foreach ($this->footnotes_buffer as $marker => $data) {
            $labels[$marker] = $data['label'];
            $names_by_marker[$marker] = implode(', ', array_keys($data['names']));
        }

        ksort($names_by_marker);

        foreach ($names_by_marker as $marker => $names_list) {
            $label = $labels[$marker] ?? 'статус не определен';
            $html .= '<p>' . esc_html($marker . ' ' . $names_list) . ' — ' . esc_html($label) . '</p>';
        }

        $html .= '</div>';
        return $html;
    }

    /**
     * Вспомогательный метод для генерации HTML-блока сносок.
     */
    public function clear_post_cache_on_save( $post_id ) {
        if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $cache_key = 'inomarker_cache_pid_' . $post_id;
        delete_transient( $cache_key . '_single' );
        delete_transient( $cache_key . '_list' );
    }

    /**
     * Очищает ВЕСЬ кэш плагина.
     * Он используется при изменении глобальных настроек.
     */
    private function clear_all_inomarker_caches() {
        global $wpdb;

        $transient_keys = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->prefix}options WHERE option_name LIKE %s",
                '\_transient\_inomarker\_cache\_pid\_%'
            )
        );

        $timeout_keys = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->prefix}options WHERE option_name LIKE %s",
                '\_transient\_timeout\_inomarker\_cache\_pid\_%'
            )
        );

        $all_keys_to_delete = array_merge($transient_keys, $timeout_keys);

        if(!empty($all_keys_to_delete)) {
            foreach ($all_keys_to_delete as $key) {
                delete_option($key);
            }
        }
    }

    /**
     * "Прокладки" для `register_settings'.
     * Они очищают кэш и возвращают чистое значение.
     */
    public function sanitize_bool_and_clear_cache( $value ) {
        $this->clear_all_inomarker_caches();
        return (bool) $value;
    }

    public function sanitize_string_and_clear_cache( $value ) {
        $this->clear_all_inomarker_caches();
        return sanitize_text_field( $value );
    }

    /**
     * Это выполняется, когда плагин деактивирован.
     * Наша задача - "навести порядок" за собой и удалить задачу из планировщика.
     */
    public static function on_deactivation() {
        wp_clear_scheduled_hook('inomarker_daily_sync');
    }

    /**
     * Гарантирует, что у нас есть только один экземпляр класса (Singleton).
     */
    public static function instance() {
        if(is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * ОТОБРАЖЕНИЕ УВЕДОМЛЕНИЙ В АДМИНКЕ
     *
     * Отображает уведомления на всех страницах админ-панели
     * если срок действия подписки истек ИЛИ вот-вот истечет.
     */
    public function show_admin_notices() {
        $statusData = get_option('inomarker_subscription_status');
        $is_plugin_enabled_by_user = get_option('inomarker_enabled', false);

        if(!$is_plugin_enabled_by_user) {
            return;
        }

        $status = $statusData['status'] ?? 'unknown';
        $settings_url = admin_url('options-general.php?page=inomarker-settings');
        $pricing_url = 'https://inomarker.ru/pricing';

        if($status !== 'active') {
            $message = $statusData['message'] ?? 'Ваша подписка Inomarker неактивна';

            echo '<div class="notice notice-error is-dismissible inomarker-notice">';
            printf(
                '<p><strong>%s</strong> %s</p>',
                '[Inomarker] Маркировка на сайте ОТКЛЮЧЕНА',
                esc_html( $message )
            );
            printf(
                '<p><a href="%s" class="button button-secondary">%s</a> <a href="%s" target="_blank" class="button button-primary">%s</a></p>',
                esc_url( $settings_url ),
                'Проверить настройки',
                esc_url( $pricing_url ),
                'Продлить подписку'
            );
            echo '</div>';
            return;
        }

        if(isset($statusData['expires_at'])) {

            $expires_timestamp = strtotime($statusData['expires_at']);
            $now_timestamp = time();

            $days_remaining = ceil(($expires_timestamp - $now_timestamp) / 86400);

            if($days_remaining > 0 && $days_remaining <= 3) {

                $days_text = $days_remaining . ' ' . (
                    ($days_remaining % 10 == 1 && $days_remaining % 100 != 11)
                        ? 'день'
                        : (($days_remaining % 10 >= 2 && $days_remaining % 10 <= 4 && ($days_remaining % 100 < 10 || $days_remaining % 100 >= 20))
                        ? 'дня'
                        : 'дней')
                    );

                $message = sprintf(
                /* translators: %s: number of days string (e.g. "5 days") */
                    'Ваша подписка Inomarker истекает. Осталось: %s',
                    $days_text
                );

                echo '<div class="notice notice-warning is-dismissible inomarker-notice">';
                printf(
                    '<p><strong>%s</strong> %s <br>%s</p>',
                    '[Иномарка] Внимание',
                    esc_html( $message ),
                    'Чтобы избежать отключения маркировки, пожалуйста, продлите подписку'
                );
                printf(
                    '<p><a href="%s" target="_blank" class="button button-primary">%s</a></p>',
                    esc_url( $pricing_url ),
                    'Продлить сейчас'
                );
                echo '</div>';
            }
        }
    }

    public function register_rest_routes() {
        register_rest_route('inomarker/v1', '/stats', [
            'methods'             => 'POST',
            'callback'            => [
                $this,
                'handle_stats_request'
            ],
            'permission_callback' => '__return_true',
        ]);
    }

    public function handle_stats_request($request) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'inomarker_stats';

        $params = $request->get_json_params();
        $url = esc_url_raw($params['url'] ?? '');
        $found = $params['found'] ?? []; // Array {name: "Name", type: "full/short"}

        if(empty($url)) return new WP_Error('no_url', 'URL is required', ['status' => 400]);

        $url_hash = $this->get_clean_url_hash( $url );
        $has_short = 0;

        // Checking for short matches
        foreach ($found as $item) {
            if(($item['type'] ?? '') === 'short') {
                $has_short = 1;
                break;
            }
        }

        // Upsert (Вставка или обновление)
        // Мы не трогаем excluded_data, мы обновляем только то, что находим.
        $existing = $wpdb->get_row($wpdb->prepare("SELECT id FROM $table_name WHERE url_hash = %s", $url_hash));// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        if($existing) {
            $wpdb->update(
                $table_name,
                [
                    'found_data'   => json_encode($found, JSON_UNESCAPED_UNICODE),
                    'has_short'    => $has_short,
                    'last_updated' => current_time('mysql')
                ],
                ['id' => $existing->id]
            );
        } else {
            $wpdb->insert(
                $table_name,
                [
                    'url_hash'      => $url_hash,
                    'url'           => $url,
                    'found_data'    => json_encode($found, JSON_UNESCAPED_UNICODE),
                    'excluded_data' => json_encode([]),
                    'has_short'     => $has_short,
                    'last_updated'  => current_time('mysql')
                ]
            );
        }

        return wp_send_json_success(['success' => true]);
    }

    public function print_footer_scripts() {
        // Если буфер пуст, это означает, что на странице ничего не было найдено, и скрипт не нужен.
        if ( empty( $this->log_buffer ) ) return;

        // Подготовка данных для JS
        $found_data = [];
        foreach ($this->log_buffer as $item) {
            // Удаление дубликатов
            $found_data[$item['name']] = [
                'name' => $item['name'],
                'type' => $item['is_short'] ? 'short' : 'full'
            ];
        }
        $found_data = array_values($found_data);

        $current_page_hash = $this->get_clean_url_hash( null );

        // 1. Регистрируем скрипт
        wp_register_script( 'inomarker-frontend', false, [], $this->version, true );

        $front_vars = [
            'foundData'  => $found_data,
            'apiUrl'     => esc_url_raw( rest_url( 'inomarker/v1/stats' ) ),
            'serverHash' => $current_page_hash,
            'nonce'      => wp_create_nonce( 'wp_rest' ),
            'wpReset'    => esc_js( wp_create_nonce( 'wp_rest' ) )
        ];

        // Создаем JS-объект InomarkerData
        wp_add_inline_script( 'inomarker-frontend', 'const InomarkerData = ' . wp_json_encode( $front_vars ) . ';' );

        $js_code = "
        (function() {
            var foundData = InomarkerData.foundData;
            var apiUrl = InomarkerData.apiUrl;

            var currentUrl = window.location.href.split('#')[0]; // Removing the hash
            var serverHash = InomarkerData.serverHash;
            var storageKey = 'inomarker_stats_' + serverHash;

            var lastSent = localStorage.getItem(storageKey);
            var now = Date.now();

            if (lastSent && (now - parseInt(lastSent)) < 86400000) {
                return;
            }

            // Shipment
            fetch(apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': InomarkerData.wpReset
                },
                body: JSON.stringify({
                    url: currentUrl,
                    found: foundData
                })
            }).then(function(res) {
                if (res.ok) {
                    localStorage.setItem(storageKey, now.toString());
                }
            });
        })();
        ";

        wp_add_inline_script( 'inomarker-frontend', $js_code );
        wp_enqueue_script( 'inomarker-frontend' );

    }

    public function render_stats_page() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'inomarker_stats';

        // Фильтры
        $search = isset($_GET['s']) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';

        $filter = isset($_GET['filter']) ? sanitize_text_field( wp_unslash( $_GET['filter'] ) ) : 'all';

        $where = "WHERE 1=1";
        if ($search) {
            $where .= $wpdb->prepare(" AND url LIKE %s", '%' . $wpdb->esc_like($search) . '%');
        }
        if ($filter === 'short') {
            $where .= " AND has_short = 1";
        }
        if ($filter === 'excluded') {
            $where .= " AND excluded_data IS NOT NULL AND excluded_data != '[]'";
        }

        // Пагинация
        $items_per_page = 20;
        $page = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
        $offset = ( $page * $items_per_page ) - $items_per_page;

        $total = $wpdb->get_var( "SELECT COUNT(id) FROM $table_name $where" );

        $items = $wpdb->get_results( "SELECT * FROM $table_name $where ORDER BY last_updated DESC LIMIT $offset, $items_per_page" );

        $total_pages = ceil($total / $items_per_page);

        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">Статистика маркировки</h1>
            <hr class="wp-header-end">

            <ul class="subsubsub">
                <li class="all"><a href="?page=inomarker-stats" class="<?php echo $filter === 'all' ? 'current' : ''; ?>">Все <span class="count">(<?php echo esc_html($total); ?>)</span></a> |</li>
                <li class="publish"><a href="?page=inomarker-stats&filter=short" class="<?php echo $filter === 'short' ? 'current' : ''; ?>">Есть "Возможно"</a> |</li>
                <li class="publish"><a href="?page=inomarker-stats&filter=excluded" class="<?php echo $filter === 'excluded' ? 'current' : ''; ?>">Есть исключения</a></li>
            </ul>

            <form method="get">
                <input type="hidden" name="page" value="inomarker-stats" />
                <p class="search-box">
                    <label class="screen-reader-text" for="search-input">Поиск:</label>
                    <input type="search" id="search-input" name="s" value="<?php echo esc_attr($search); ?>">
                    <input type="submit" id="search-submit" class="button" value="Поиск URL">
                </p>
            </form>

            <table class="wp-list-table widefat fixed striped table-view-list posts">
                <thead>
                <tr>
                    <th>URL Страницы</th>
                    <th style="width: 100px;">Кол-во</th>
                    <th>Найденные лица (Кликните, чтобы исключить)</th>
                    <th style="width: 150px;">Дата</th>
                    <th style="width: 50px;"></th> </tr>
                </tr>
                </thead>
                <tbody>
                <?php if (empty($items)): ?>
                    <tr><td colspan="4">Данных пока нет</td></tr>
                <?php else: ?>
                    <?php foreach ($items as $item):
                        $found = json_decode($item->found_data, true) ?? [];
                        $excluded = json_decode($item->excluded_data, true) ?? [];
                        ?>
                        <tr>
                            <td>
                                <a href="<?php echo esc_url($item->url); ?>" target="_blank">
                                    <strong><?php echo esc_html(substr($item->url, strpos($item->url, '://') + 3)); ?></strong>
                                </a>
                                <?php if ($item->has_short): ?>
                                    <span style="background:#fef3c7; color:#92400e; font-size:10px; padding:2px 5px; border-radius:3px; margin-left:5px;">Возможно</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo count($found); ?></td>
                            <td>
                                <div class="im-tags-list">
                                    <?php foreach ($found as $entity):
                                        $name = $entity['name'];
                                        $is_excluded = in_array($name, $excluded);
                                        $is_short = ($entity['type'] === 'short');
                                        $class = $is_excluded ? 'im-tag-excluded' : 'im-tag-active';
                                        $title_text = $is_excluded ? 'Вернуть в маркировку' : 'Исключить из маркировки';
                                        if ($is_short) $class .= ' im-tag-short';
                                        ?>
                                        <button type="button"
                                                class="im-tag <?php echo esc_attr($class); ?>"
                                                onclick="toggleExclusion(this, <?php echo (int) $item->id; ?>, '<?php echo esc_js( $name ); ?>')"
                                                title="<?php echo esc_attr($title_text); ?>">
                                            <?php echo esc_html($name); ?>
                                            <?php if ($is_excluded): ?><span class="dashicons dashicons-no"></span><?php endif; ?>
                                        </button>
                                    <?php endforeach; ?>
                                </div>
                            </td>
                            <td><?php echo esc_html( date_i18n( 'd.m.Y H:i', strtotime( $item->last_updated ) ) ); ?></td>
                            <td>
                                <button type="button"
                                        class="button-link-delete"
                                        onclick="deletePageStat(this, <?php echo (int) $item->id; ?>)"
                                        title="Удалить статистику страницы">
                                    <span class="dashicons dashicons-trash"></span>
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>

            <div class="tablenav bottom">
                <div class="tablenav-pages">
                    <?php if($page > 1): ?><a class="button" href="?page=inomarker-stats&paged=<?php echo esc_attr($page-1); ?>&filter=<?php echo esc_attr($filter); ?>">&laquo;</a><?php endif; ?>
                    <span class="paging-input"><?php echo esc_html($page); ?> / <?php echo esc_html($total_pages); ?></span>
                    <?php if($page < $total_pages): ?><a class="button" href="?page=inomarker-stats&paged=<?php echo esc_attr($page+1); ?>&filter=<?php echo esc_attr($filter); ?>">&raquo;</a><?php endif; ?>
                </div>
            </div>
        </div>

        <?php
    }

    public function ajax_toggle_exclusion() {
        check_ajax_referer( 'inomarker_exclude' );

        if ( ! current_user_can( 'manage_options' ) ) wp_die();

        if ( empty( $_POST['id'] ) || empty( $_POST['name'] ) ) {
            wp_send_json_error( 'Отсутствуют необходимые параметры' );
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'inomarker_stats';
        $id = intval( wp_unslash( $_POST['id'] ) );
        $name = sanitize_text_field( wp_unslash( $_POST['name'] ) );

        $row = $wpdb->get_row( $wpdb->prepare( "SELECT excluded_data FROM $table_name WHERE id = %d", $id ) );
        if ( ! $row ) wp_send_json_error();

        $excluded = json_decode( $row->excluded_data, true ) ?? [];

        $status = '';
        if ( in_array( $name, $excluded ) ) {
            // Исключение из исключений (в том числе)
            $excluded = array_diff( $excluded, [$name] );
            $status = 'active';
        } else {
            // Добавить в исключения (отключить)
            $excluded[] = $name;
            $status = 'excluded';
        }

        // Сохранение и обновление даты для сброса кэша (если он был)
        $wpdb->update(
            $table_name,
            [ 'excluded_data' => wp_json_encode( array_values($excluded), JSON_UNESCAPED_UNICODE ) ],
            [ 'id' => $id ]
        );

        $this->clear_all_inomarker_caches();

        wp_send_json_success( ['status' => $status] );
    }

    /**
     * AJAX: Удаляет страницу из статистики.
     */
    public function ajax_delete_stat() {
        check_ajax_referer( 'inomarker_delete' ); // Проверка безопасности

        if ( ! current_user_can( 'manage_options' ) ) wp_die();

        if ( empty( $_POST['id'] ) ) {
            wp_send_json_error( 'Отсутствующий идентификатор' );
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'inomarker_stats';
        $id = intval( wp_unslash( $_POST['id'] ) );

        // Удаление записи
        $result = $wpdb->delete( $table_name, [ 'id' => $id ] );

        $this->clear_all_inomarker_caches();

        if ( $result ) {
            wp_send_json_success();
        } else {
            wp_send_json_error( 'Не удалось удалить запись или она уже удалена' );
        }
    }

    /**
     * Нормализует URL-адрес для создания стабильного хэша.
     */
    private function get_clean_url_hash( $url ) {
        // Если URL-адрес не передан, мы собираем текущий URL-адрес
        if ( empty( $url ) ) {
            global $wp;
            // home_url( $wp->request ) выдает URL-адрес без параметров.
            // Для ?p=123 нам нужен add_query_arg, чтобы восстановить их.,
            // или мы берем "сырой" REQUEST_URI, но это опасно.
            // Самый безопасный способ - это использовать полный текущий URL-адрес следующим образом:
//            $url = set_url_scheme( 'http://' . wp_unslash( $_SERVER['HTTP_HOST'] ) . wp_unslash( $_SERVER['REQUEST_URI'] ) );
            if ( empty( $url ) ) {
                $host = '';
                if ( isset( $_SERVER['HTTP_HOST'] ) ) {
                    // Защита от внедрения заголовка хоста
                    $host = sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) );
                }

                $request_uri = '';
                if ( isset( $_SERVER['REQUEST_URI'] ) ) {
                    $request_uri = sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) );
                }

                $url = set_url_scheme( 'http://' . $host . $request_uri );
            }
        }

        $parts = wp_parse_url( $url );
        $host  = isset($parts['host']) ? str_replace( 'www.', '', strtolower( $parts['host'] ) ) : '';
        $path  = isset($parts['path']) ? rtrim( $parts['path'], '/' ) : '';
        $query = $parts['query'] ?? '';

        $final_query = '';

        if ( ! empty( $query ) ) {
            parse_str( $query, $params );

            // Список "важных" настроек WordPress, которые определяют контент.
            // Остальные (utm_*, fbclid, ref и т.д.) будут отброшены.
            $whitelist = [
                'p',          // ID поста (?p=123)
                'page_id',    // Page ID (?page_id=2)
                'cat',        // Category ID
                'tag',        // The tag
                'author',     // Author
                'post_type',  // Type of post
                's',          // Search query (?s=search)
                'attachment_id', // Media
            ];

            $keep_params = [];
            foreach ( $whitelist as $key ) {
                if ( isset( $params[ $key ] ) ) {
                    $keep_params[ $key ] = $params[ $key ];
                }
            }

            // Если есть важные параметры, добавьте их в хэш.
            if ( ! empty( $keep_params ) ) {
                // Сортировка таким образом, чтобы ?было ли p=1иa=b равно ?a=b и p=1 (хотя мы уже выбросили 'a')
                ksort( $keep_params );
                $final_query = '?' . http_build_query( $keep_params );
            }
        }

        // Собираем строку: "site.com/page " или "site.com/?p=123 "
        $clean_string = $host . $path . $final_query;

        return md5( $clean_string );
    }

    /**
     * Проверяет, активирован ли режим отладки через URL.
     * Формат: ?im_debug=MD5(API_KEY)
     */
    private function is_debug_active() {
        if ( ! isset( $_GET['im_debug'] ) ) {
            return false;
        }

        $api_key = get_option( 'inomarker_api_key' );
        if ( empty( $api_key ) ) {
            return false;
        }

        // Сравниваем переданный хеш с md5 от реального ключа
        // Это безопасно, так как мы не светим сам ключ, а хеш подобрать сложно.
        return sanitize_text_field( $_GET['im_debug'] ) === md5( $api_key );
    }

    /**
     * Проверка обновлений на удаленном сервере.
     */
    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        $remote = $this->get_remote_info();

        if ($remote && version_compare($this->version, $remote->version, '<')) {
            $res = new stdClass();
            $res->slug = 'inomarker'; // Папка плагина
            $res->plugin = 'inomarker/inomarker.php'; // Путь к главному файлу
            $res->new_version = $remote->version;
            $res->tested = $remote->tested;
            $res->package = $remote->download_url;

            $transient->response[$res->plugin] = $res;
        }

        return $transient;
    }

    /**
     * Получение данных о плагине для окна "Посмотреть детали".
     */
    public function plugin_popup_info($res, $action, $args) {
        if ($action !== 'plugin_information') return $res;
        if ($args->slug !== 'inomarker') return $res;

        $remote = $this->get_remote_info();

        if ($remote) {
            $res = new stdClass();
            $res->name = $remote->name;
            $res->slug = 'inomarker';
            $res->version = $remote->version;
            $res->tested = $remote->tested;
            $res->requires = $remote->requires;
            $res->author = 'Inomarker Team';
            $res->download_link = $remote->download_url;
            $res->sections = [
                'description' => $remote->sections->description,
                'changelog'   => $remote->sections->changelog
            ];
            return $res;
        }

        return $res;
    }

    /**
     * Вспомогательный метод для связи с вашим API.
     */
    private function get_remote_info() {
        // Используем transient, чтобы не дергать ваш сервер при каждой загрузке страницы админки
        $remote = get_transient('inomarker_update_info');

        if (false === $remote) {
            $response = wp_remote_get('https://inomarker.ru/api/v1/plugin/info', [
                'timeout' => 10,
                'headers' => ['Accept' => 'application/json']
            ]);

            if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                return false;
            }

            $remote = json_decode(wp_remote_retrieve_body($response));
            set_transient('inomarker_update_info', $remote, HOUR_IN_SECONDS);
        }

        return $remote;
    }

}

/**
 * Функция глобального загрузчика.
 * @return Inomarker_Plugin
 */
function Inomarker() {
    return Inomarker_Plugin::instance();
}

// -------------------------------------------------
// ЗАПУСК ПЛАГИНА
// -------------------------------------------------

// Мы регистрируем наши крючки для активации и деактивации
register_activation_hook(__FILE__, [
    'Inomarker_Plugin',
    'on_activation'
]);
register_deactivation_hook(__FILE__, [
    'Inomarker_Plugin',
    'on_deactivation'
]);

// Запуск плагина
Inomarker();
